/* wmkitt.c: a dockable KITT scanner applet for Window Maker
 *
 * KITT portions by Scott Jann <sjann@knight-rider.org>.
 * Copyright Scott Jann
 * 
 * Significant portions of this software are derived from asclock by
 * Beat Christen <spiff@longstreet.ch>.  Such portions are copyright
 * by Beat Christen and the other authors of asclock.
 *
 * The rest is mostly from wsclock by by Jim Knoble <jmknoble@pobox.com>.
 * wsclock is copyright Jim Knoble.
 * 
 * Disclaimer:
 * 
 * The software is provided "as is", without warranty of any kind,
 * express or implied, including but not limited to the warranties of
 * merchantability, fitness for a particular purpose and
 * noninfringement. In no event shall the author(s) be liable for any
 * claim, damages or other liability, whether in an action of
 * contract, tort or otherwise, arising from, out of or in connection
 * with the software or the use or other dealings in the software.
 */

#include <sys/types.h>
#include <sys/wait.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <X11/Xatom.h>
#include <X11/Xlib.h>
#include <X11/xpm.h>
#include <X11/extensions/shape.h>

#include "xpm/mask.xbm"
#include "xpm/mask.xpm"

typedef enum BlipState {
	blipPreLight,
	blipOn,
	blipDimming,
	blipOff
}
BlipState;

#define OUR_WINDOW_EVENTS	(ExposureMask | ButtonPressMask | StructureNotifyMask)
#define DEFAULT_XPM_CLOSENESS	40000
#define SCANNER_WIDTH		52
#define BLIP_WIDTH		2
#define BLIP_HEIGHT		7
#define OFFSET_X		2
#define OFFSET_Y		25
#define NUM_BLIPS		(SCANNER_WIDTH / BLIP_WIDTH)
#define NUM_COLORS		48
#define PRE_LIGHT_JUMP		4
#define ON_LENGTH		PRE_LIGHT_JUMP

XColor		red_colors[NUM_COLORS];
XColor		yellow_colors[NUM_COLORS];
XColor		*colors = red_colors;

BlipState	states[NUM_BLIPS];
int		counters[NUM_BLIPS];

int		iterator = 0;
int		direction = 0;
int		evil = 0;


Display		*dpy;
Window		rootWindow;
int		screen;
int		displayDepth;
XSizeHints	sizeHints;
XWMHints	wmHints;
Pixel		bgPixel, fgPixel;
GC		normalGC;
Window		iconWin, win;

char		*progName;
char		*className = "WMKITT";
char		*geometry = "";

char *errColorCells = "not enough free color cells or xpm not found\n";

typedef struct _XpmIcon {
    Pixmap        pixmap;
    Pixmap        mask;
    XpmAttributes attributes;
} XpmIcon;

XpmIcon kittBg;
XpmIcon visible;

char *usageText[] = {
"Options:",
"    -evil                   show evil scanner",
NULL
};

char *version = VERSION;

/**********************************************************************/
/* Display usage information */
void showUsage()
{
   char **cpp;
   
   fprintf(stderr, "Usage:  %s [option [option ...]]\n\n", progName);
   for (cpp = usageText; *cpp; cpp++)
    {
       fprintf(stderr, "%s\n", *cpp);
    }
   fprintf(stderr,"\n");
   exit(1);
}

/* Display the program version */
void showVersion()
{
   fprintf(stderr, "%s version %s\n", progName, version);
   exit(1);
}

/* Display an error message */
void showError(char *message, char *data)
{
   fprintf(stderr,"%s: can't %s %s\n", progName, message, data);
}

/* Display an error message and exit */
void showFatalError(char *message, char *data)
{
   showError(message, data);
   exit(1);
}

/* Konvertiere XPMIcons nach Pixmaps */
void GetXpms(void)
{
   XWindowAttributes attributes;
   int               status;
   
   /* for the colormap */
   XGetWindowAttributes(dpy, rootWindow, &attributes);

   kittBg.attributes.closeness = DEFAULT_XPM_CLOSENESS;
   kittBg.attributes.valuemask |=
      (XpmReturnPixels | XpmReturnExtensions | XpmCloseness);
   
       status = XpmCreatePixmapFromData(dpy, rootWindow, mask_xpm,
					&kittBg.pixmap, &kittBg.mask,
					&kittBg.attributes);
   if (XpmSuccess != status)
    {
       showFatalError("create pixmap:", errColorCells);
    }

   visible.attributes.depth  = displayDepth;
   visible.attributes.width  = kittBg.attributes.width;
   visible.attributes.height = kittBg.attributes.height;
   visible.pixmap = XCreatePixmap(dpy, rootWindow, visible.attributes.width,
				  visible.attributes.height,
				  visible.attributes.depth);
}

/* Remove expose events for a specific window from the queue */
int flushExposeEvents(Window w)
{
   XEvent dummy;
   int    i = 0;
   
   while (XCheckTypedWindowEvent(dpy, w, Expose, &dummy))
    {
       i++;
    }
   return(i);
}

/* (Re-)Draw the main window and the icon window */
void redrawWindow(XpmIcon *v)
{
   flushExposeEvents(iconWin);
   XCopyArea(dpy, v->pixmap, iconWin, normalGC,
	     0, 0, v->attributes.width, v->attributes.height, 0, 0);
   flushExposeEvents(win);
   XCopyArea(dpy, v->pixmap, win, normalGC,
	     0, 0, v->attributes.width, v->attributes.height, 0, 0);
}

/* Get a Pixel for the given color name */
Pixel GetColor(char *colorName)
{
   XColor            color;
   XWindowAttributes attributes;
   
   XGetWindowAttributes(dpy, rootWindow, &attributes);
   color.pixel = 0;
   if (!XParseColor(dpy, attributes.colormap, colorName, &color)) 
    {
       showError("parse color", colorName);
    }
   else if (!XAllocColor(dpy, attributes.colormap, &color)) 
    {
       showError("allocate color", colorName);
    }
   return(color.pixel);
}


void initColors()
{
	int i;
	XWindowAttributes attributes;

	/* for the colormap */
	XGetWindowAttributes(dpy, rootWindow, &attributes);

	for(i = 0; i < NUM_COLORS; i++) {
		red_colors[i].flags = DoRed | DoGreen | DoBlue;
		red_colors[i].red = 0xFFFF - i * 0xFFFF / NUM_COLORS;
		red_colors[i].green = 0;
		red_colors[i].blue = 0;
		if (!XAllocColor(dpy, attributes.colormap, &red_colors[i])) 
			showError("allocate color", "red");
	}
	for(i = 0; i < NUM_COLORS; i++) {
		yellow_colors[i].flags = DoRed | DoGreen | DoBlue;
		yellow_colors[i].red = 0xFFFF - i * 0xFFFF / NUM_COLORS;
		yellow_colors[i].green = 0xFFFF - i * 0xFFFF / NUM_COLORS;
		yellow_colors[i].blue = 0;
		if (!XAllocColor(dpy, attributes.colormap, &yellow_colors[i])) 
			showError("allocate color", "yellow");
	}

	for(i = 0; i < NUM_BLIPS; i++) {
		states[i] = blipOff;
		counters[i] = 0;
	}
	if(evil)
		colors = yellow_colors;
}

/* pos = 0 to NUM_BLIPS - 2 */
/* color = 0 to NUM_COLORS - 1 */
void showBlip(int pos, int color)
{
	XSetForeground(dpy, normalGC, colors[color % NUM_COLORS].pixel);
	XFillRectangle(dpy, visible.pixmap, normalGC,
		OFFSET_X + pos * BLIP_WIDTH, OFFSET_Y,
		BLIP_WIDTH, BLIP_HEIGHT);
}

int getBlipColor(int i)
{
	switch(states[i])
	{
	case blipOn:
		return(0);
	case blipPreLight:
	case blipDimming:
		return(counters[i]);
	blipOff:
	default:
		return(NUM_COLORS - 1);
	}
}

void incrementBlips()
{
	int i;

	for(i = 0; i < NUM_BLIPS; i++) {
		switch(states[i]) {
		case blipOn:
			counters[i]++;
			if(counters[i] >= ON_LENGTH)
			{
				states[i] = blipDimming;
				counters[i] = 0;
			}
			break;
		case blipPreLight:
			counters[i] -= PRE_LIGHT_JUMP;
			if(counters[i] < PRE_LIGHT_JUMP)
			{
				states[i] = blipOn;
				counters[i] = 0;
			}
			break;
		case blipDimming:
			counters[i]++;
			if(counters[i] >= NUM_COLORS)
			{
				states[i] = blipOff;
				counters[i] = 0;
			}
			break;
		case blipOff: /* terminal */
		default:
			states[i] = blipOff;
			counters[i] = 0;
			break;
		}
	}
}

void showScanner()
{
	int i;
	int lit = 0;
	incrementBlips();

   for(i = 0; i < NUM_BLIPS; i++) {
	   if(states[i] == blipOn || states[i] == blipPreLight)
		   lit++;
   }

   if(lit == 0)
   {
	   if(direction == 0)
	   {
		   direction = 1;
		   iterator = NUM_BLIPS - 1;
	   } else {
		   direction = 0;
		   iterator = 0;
	   }
   }

   if(iterator > -1 && iterator < NUM_BLIPS)
   {
	   if(states[iterator] == blipOn)
	   {
		   counters[iterator] = 0;
	   } else {
		   if(states[iterator] != blipDimming)
			   counters[iterator] = NUM_COLORS - 1;
		   states[iterator] = blipPreLight;
	   }
   }

   for(i = 0; i < NUM_BLIPS; i++) {
	   showBlip(i, getBlipColor(i));
   }


   if(direction == 0)
	   iterator++;
   else
	   iterator--;
}

void updateWindow()
{
   XCopyArea(dpy, kittBg.pixmap, visible.pixmap, normalGC,
	     0, 0, sizeHints.width, sizeHints.height, 0, 0);

   showScanner();
}

/* Extract program name from the zeroth program argument */
char *extractProgName(char *argv0)
{
   char *prog_name = NULL;
   
   if (NULL != argv0)
    {
       prog_name = strrchr(argv0, '/');
       if (NULL == prog_name)
	{
	   prog_name = argv0;
	}
       else
	{
	   prog_name++;
	}
    }
   return(prog_name);
}

/* Process program arguments and set corresponding options */
int processArgs(int argc, char **argv)
{
   int i;
   
   for (i = 1; i < argc; i++)
    {
       if (0 == strcmp(argv[i], "--"))
	{
	   break;
	}
       else if ((0 == strcmp(argv[i], "-evil")) ||
		(0 == strcmp(argv[i], "-e")) ||
		(0 == strcmp(argv[i], "--evil")))
	{
	   evil = 1;
	}
       else if ((0 == strcmp(argv[i], "-version")) ||
		(0 == strcmp(argv[i], "-V")) ||
		(0 == strcmp(argv[i], "--version")))
	{
	   showVersion();
	}
       else if ((0 == strcmp(argv[i], "-help")) ||
		(0 == strcmp(argv[i], "-h")) ||
		(0 == strcmp(argv[i], "--help")))
	{
	   showUsage();
	}
       else
	{
	   fprintf(stderr, "%s: unrecognized option `%s'\n",
		   progName, argv[i]);
	   showUsage();
	}
    }
   return(i);
}

/**********************************************************************/
int main(int argc, char **argv)
{
   int           i;
   unsigned int  borderWidth = 0;
   char          *displayName = NULL; 
   XGCValues     gcValues;
   unsigned long gcMask;
   XEvent        event;
   XTextProperty wmName;
   XClassHint    classHint;
   Pixmap        shapeMask;
   
   /* Parse command line options */
   progName = extractProgName(argv[0]);
   processArgs(argc, argv);
   
   /* Open the display */
   dpy = XOpenDisplay(displayName);
   if (NULL == dpy)  
    { 
       fprintf(stderr, "%s: can't open display %s\n", progName,
	       XDisplayName(displayName));
       exit(1); 
    }
   screen       = DefaultScreen(dpy);
   rootWindow   = RootWindow(dpy, screen);
   displayDepth = DefaultDepth(dpy, screen);
   /* xFd          = XConnectionNumber(dpy); */
   
   /* Icon Daten nach XImage konvertieren */
   initColors();
   GetXpms();
   
   /* Create a window to hold the banner */
   sizeHints.x = 0;
   sizeHints.y = 0;
   sizeHints.min_width  = kittBg.attributes.width;
   sizeHints.min_height = kittBg.attributes.height;
   sizeHints.max_width  = kittBg.attributes.width;
   sizeHints.max_height = kittBg.attributes.height;
   sizeHints.base_width  = kittBg.attributes.width;
   sizeHints.base_height = kittBg.attributes.height;
   sizeHints.flags = USSize | USPosition | PMinSize | PMaxSize | PBaseSize;
   
   bgPixel = GetColor("white");
   fgPixel = GetColor("black");
   
   XWMGeometry(dpy, screen, geometry, NULL, borderWidth, &sizeHints,
	       &sizeHints.x, &sizeHints.y, &sizeHints.width, &sizeHints.height,
	       &sizeHints.win_gravity);
   sizeHints.width  = kittBg.attributes.width;
   sizeHints.height = kittBg.attributes.height;
   
   win = XCreateSimpleWindow(dpy, rootWindow, sizeHints.x, sizeHints.y,
			     sizeHints.width, sizeHints.height,
			     borderWidth, fgPixel, bgPixel);
   iconWin = XCreateSimpleWindow(dpy, win, sizeHints.x, sizeHints.y,
				 sizeHints.width, sizeHints.height,
				 borderWidth, fgPixel, bgPixel);

   /* Hints aktivieren */
   XSetWMNormalHints(dpy, win, &sizeHints);
   classHint.res_name = progName;
   classHint.res_class = className;
   XSetClassHint(dpy, win, &classHint);
   
   XSelectInput(dpy, win, OUR_WINDOW_EVENTS);
   XSelectInput(dpy, iconWin, OUR_WINDOW_EVENTS);
   
   if (0 == XStringListToTextProperty(&progName, 1, &wmName))
    {
       fprintf(stderr, "%s: can't allocate window name text property\n",
	       progName);
       exit(-1);
    }
   XSetWMName(dpy, win, &wmName);
  
   /* Create a GC for drawing */
   gcMask = GCForeground | GCBackground | GCGraphicsExposures;
   gcValues.foreground = fgPixel;
   gcValues.background = bgPixel;
   gcValues.graphics_exposures = False;
   normalGC = XCreateGC(dpy, rootWindow, gcMask, &gcValues);

   shapeMask = XCreateBitmapFromData(dpy, win, (char *)mask_bits,
					 mask_width, mask_height);
   XShapeCombineMask(dpy, win, ShapeBounding, 0, 0, shapeMask, ShapeSet);
   XShapeCombineMask(dpy, iconWin, ShapeBounding, 0, 0, shapeMask,
			 ShapeSet);
  
   wmHints.initial_state = WithdrawnState;
   wmHints.icon_window = iconWin;
   wmHints.icon_x = sizeHints.x;
   wmHints.icon_y = sizeHints.y;
   wmHints.window_group = win;
   wmHints.flags = StateHint | IconWindowHint | IconPositionHint |
      WindowGroupHint;
   XSetWMHints(dpy, win, &wmHints);

   XSetCommand(dpy, win, argv, argc);
   XMapWindow(dpy,win);

   updateWindow();
   redrawWindow(&visible);
   while (1)
    {
       updateWindow();
       redrawWindow(&visible);
       
       /* read a packet */
       while (XPending(dpy))
	{
	   XNextEvent(dpy, &event);
	   switch(event.type)
	    {
	     case Expose:
	       if (0 == event.xexpose.count)
		{
		   redrawWindow(&visible);
		}
	       break;
	    case ButtonPress:
		    if(colors == red_colors)
			    colors = yellow_colors;
		    else
			    colors = red_colors;
	       break;
	    case DestroyNotify:
	       XFreePixmap(dpy, visible.pixmap);
	       XCloseDisplay(dpy);
	       exit(0); 
	     default:
	       break;      
	    }
	}
       XFlush(dpy);
#ifdef SYSV
	   poll((struct poll *) 0, (size_t) 0, 10);	/* 1/5 sec */
#else
	   usleep(10000L);	/* 1/5 sec */
#endif
    }
   return(0);
}

