using System;
using System.Net;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using System.Text;
using System.Text.RegularExpressions;
using System.Diagnostics;

namespace SystemInfo
{
	class SystemInfo
	{
		private static bool m_showTimeZone = true;
		private readonly static string m_timeFormatRapid = "G";
		private readonly static string m_timeFormatSlow = "g";
		private static PerformanceCounter cpuCounter = new PerformanceCounter("Processor", "% Processor Time", "_Total");
		private static PerformanceCounter diskCounter = new PerformanceCounter("LogicalDisk", "% Free Space", "_Total");
		private static PerformanceCounter ramCounter = new PerformanceCounter("Memory", "Available MBytes");

		private static string Abbreviate(string str)
		{
			Regex r = new Regex(@"[a-z\s]");
			return(r.Replace(str, ""));
		}

		public static bool ShowTimeZone
		{
			get { return (m_showTimeZone); }
			set { m_showTimeZone = value; }
		}

		public static string IPAddress
		{
			get
			{
				string result = "";
				IPHostEntry ipEntry = Dns.GetHostEntry(Dns.GetHostName());
				foreach (IPAddress ip in ipEntry.AddressList)
				{
					if (String.IsNullOrEmpty(result) == false)
						result += ", ";
					result += ip.ToString();
				}
				return (result);
			}
		}

		public static string ProcessorUsage
		{
			get
			{
				string result = cpuCounter.NextValue().ToString("F00") + "%";
				return (result);
			}
		}

		public static string DiskUsage
		{
			get
			{
				string result = diskCounter.NextValue().ToString("F01") + "% Free";
				return (result);
			}
		}

		public static string MemoryUsage
		{
			get
			{
				string result = ramCounter.NextValue() + "MB Free";
				return (result);
			}
		}

		public static string ProcessCount
		{
			get
			{
				string result = Process.GetProcesses().Length.ToString();
				return (result);
			}
		}

		public static string UserDomain
		{
			get
			{
				string result = "";
				if (String.IsNullOrEmpty(Environment.UserDomainName) == false)
					result = Environment.UserDomainName + "\\";
				result += Environment.UserName;
				return (result);
			}
		}

		public static string MachineName
		{
			get
			{
				return (Environment.MachineName);
			}
		}

		public static string Time
		{
			get
			{
				DateTime now = DateTime.Now;
				string timeFormat = Properties.Settings.Default.UpdateFreq > 500 ? m_timeFormatSlow : m_timeFormatRapid;
				string result = now.ToString(timeFormat);
				if (m_showTimeZone)
				{
					string timeZone = Abbreviate(TimeZone.CurrentTimeZone.IsDaylightSavingTime(now) ? TimeZone.CurrentTimeZone.DaylightName : TimeZone.CurrentTimeZone.StandardName);
					result = String.Format("{0} {1} ({2})", now.ToString(timeFormat), timeZone, TimeZone.CurrentTimeZone.GetUtcOffset(now).TotalHours);
				}
				return (result);
			}
		}

		public static string OSVersion
		{
			get
			{
				string result = "";

				switch (Environment.OSVersion.Platform)
				{
					case PlatformID.Win32NT:
						if (Environment.OSVersion.Version.Major <= 4)
							result = String.Format("Windows NT {0}.{1}", Environment.OSVersion.Version.Major, Environment.OSVersion.Version.Minor);
						else if (Environment.OSVersion.Version.Major == 5 && Environment.OSVersion.Version.Minor == 0)
							result = "Windows 2000";
						else if (Environment.OSVersion.Version.Major == 5 && Environment.OSVersion.Version.Minor == 1)
							result = "Windows XP";
						else if (Environment.OSVersion.Version.Major == 5 && Environment.OSVersion.Version.Minor == 2)
							result = "Windows 2003";
						else if (Environment.OSVersion.Version.Major == 6 && Environment.OSVersion.Version.Minor == 0)
							result = "Windows Vista";
						break;
					case PlatformID.Win32Windows:
						if (Environment.OSVersion.Version.Minor == 0)
							result = "Windows 95";
						else if (Environment.OSVersion.Version.Minor == 10)
							result = "Windows 98";
						else if (Environment.OSVersion.Version.Minor == 90)
							result = "Windows Me";
						break;
					case PlatformID.WinCE:
						result = String.Format("Windows CE {0}.{1}", Environment.OSVersion.Version.Major, Environment.OSVersion.Version.Minor);
						break;
					case PlatformID.Win32S:
					case PlatformID.Unix:
					default:
						result = String.Format("Windows {0}.{1}", Environment.OSVersion.Version.Major, Environment.OSVersion.Version.Minor);
						break;
				}

				if (String.IsNullOrEmpty(Environment.OSVersion.ServicePack) == false)
					result += " " + Abbreviate(Environment.OSVersion.ServicePack);
				return (result);
			}
		}
	}

	// Class to assist with Win32 API calls
	class APIHelp
	{
		// Required constants
		public const Int32 WS_EX_LAYERED = 0x80000;
		public const Int32 HTCAPTION = 0x02;
		public const Int32 WM_NCHITTEST = 0x84;
		public const Int32 WM_NCLBUTTONDBLCLK = 0xa3;
		public const Int32 WM_ACTIVEATE = 0x0006;
		public const Int32 ULW_ALPHA = 0x02;
		public const byte AC_SRC_OVER = 0x00;
		public const byte AC_SRC_ALPHA = 0x01;
		public const int HWND_BOTTOM = 0x1;
		public const uint SWP_NOSIZE = 0x1;
		public const uint SWP_NOMOVE = 0x2;
		public const uint SWP_SHOWWINDOW = 0x40;

		public enum Bool
		{
			False = 0,
			True = 1
		}

		[StructLayout(LayoutKind.Sequential)]
		public struct Point
		{
			public Int32 x;
			public Int32 y;

			public Point(Int32 x, Int32 y) { this.x = x; this.y = y; }
		}

		[StructLayout(LayoutKind.Sequential)]
		public struct Size
		{
			public Int32 cx;
			public Int32 cy;

			public Size(Int32 cx, Int32 cy) { this.cx = cx; this.cy = cy; }
		}

		[StructLayout(LayoutKind.Sequential, Pack = 1)]
		struct ARGB
		{
			public byte Blue;
			public byte Green;
			public byte Red;
			public byte Alpha;
		}

		[StructLayout(LayoutKind.Sequential, Pack = 1)]
		public struct BLENDFUNCTION
		{
			public byte BlendOp;
			public byte BlendFlags;
			public byte SourceConstantAlpha;
			public byte AlphaFormat;
		}

		[DllImport("user32.dll", ExactSpelling = true, SetLastError = true)]
		public static extern Bool UpdateLayeredWindow(IntPtr hwnd, IntPtr hdcDst, ref Point pptDst, ref Size psize, IntPtr hdcSrc, ref Point pprSrc, Int32 crKey, ref BLENDFUNCTION pblend, Int32 dwFlags);

		[DllImport("gdi32.dll", ExactSpelling = true, SetLastError = true)]
		public static extern IntPtr CreateCompatibleDC(IntPtr hDC);

		[DllImport("user32.dll", ExactSpelling = true, SetLastError = true)]
		public static extern IntPtr GetDC(IntPtr hWnd);

		[DllImport("user32.dll", ExactSpelling = true)]
		public static extern int ReleaseDC(IntPtr hWnd, IntPtr hDC);

		[DllImport("gdi32.dll", ExactSpelling = true, SetLastError = true)]
		public static extern Bool DeleteDC(IntPtr hdc);

		[DllImport("gdi32.dll", ExactSpelling = true)]
		public static extern IntPtr SelectObject(IntPtr hDC, IntPtr hObject);

		[DllImport("gdi32.dll", ExactSpelling = true, SetLastError = true)]
		public static extern Bool DeleteObject(IntPtr hObject);

		[DllImport("user32.dll", EntryPoint = "SetWindowPos")]
		public static extern bool SetWindowPos(
		   IntPtr hWnd,               // window handle
		   IntPtr hWndInsertAfter,    // placement-order handle
		   int X,                  // horizontal position
		   int Y,                  // vertical position
		   int cx,                 // width
		   int cy,                 // height
		   uint uFlags);           // window positioning flags
	}
}
