using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Text;
using System.Drawing.Imaging;
using System.Text;
using System.Windows.Forms;

namespace SystemInfo
{
	public partial class Form1 : Form
	{
		private readonly int m_screenOffset = 25;
		private readonly int m_shadowDepth = 1;
		private readonly string m_font = "Arial";
		private readonly float m_fontSize = 14f;
		private readonly Color m_shadowColor = Color.FromArgb(60, 0, 0, 0);
		private bool m_started = false;

		public Form1()
		{
			InitializeComponent();
		}

		// Let Windows drag this window for us (thinks its hitting the title bar of the window)
		protected override void WndProc(ref Message message)
		{
			if (message.Msg == APIHelp.WM_NCHITTEST)
			{
				// Tell Windows that the user is on the title bar (caption)
				message.Result = (IntPtr)APIHelp.HTCAPTION;
				SendBehind();
			}
			else if (message.Msg == APIHelp.WM_NCLBUTTONDBLCLK)
			{
				// do nothing on title bar double click
			}
			else
			{
				base.WndProc(ref message);
			}
		}

		#region Transparent window support
		// http://blogs.msdn.com/mswanson/archive/2005/07/07/436618.aspx
		protected override CreateParams CreateParams
		{
			get
			{
				// Add the layered extended style (WS_EX_LAYERED) to this window
				CreateParams createParams = base.CreateParams;
				createParams.ExStyle |= APIHelp.WS_EX_LAYERED;
				return createParams;
			}
		}

		public void SelectBitmap(Bitmap bitmap)
		{
			// Does this bitmap contain an alpha channel?
			if (bitmap.PixelFormat != PixelFormat.Format32bppArgb)
			{
				throw new ApplicationException("The bitmap must be 32bpp with alpha-channel.");
			}

			// Get device contexts
			IntPtr screenDc = APIHelp.GetDC(IntPtr.Zero);
			IntPtr memDc = APIHelp.CreateCompatibleDC(screenDc);
			IntPtr hBitmap = IntPtr.Zero;
			IntPtr hOldBitmap = IntPtr.Zero;

			try
			{
				// Get handle to the new bitmap and select it into the current device context
				hBitmap = bitmap.GetHbitmap(Color.FromArgb(0));
				hOldBitmap = APIHelp.SelectObject(memDc, hBitmap);

				// Set parameters for layered window update
				APIHelp.Size newSize = new APIHelp.Size(bitmap.Width, bitmap.Height);	// Size window to match bitmap
				APIHelp.Point sourceLocation = new APIHelp.Point(0, 0);
				APIHelp.Point newLocation = new APIHelp.Point(this.Left, this.Top);		// Same as this window
				APIHelp.BLENDFUNCTION blend = new APIHelp.BLENDFUNCTION();
				blend.BlendOp = APIHelp.AC_SRC_OVER;						// Only works with a 32bpp bitmap
				blend.BlendFlags = 0;											// Always 0
				blend.SourceConstantAlpha = 255;										// Set to 255 for per-pixel alpha values
				blend.AlphaFormat = APIHelp.AC_SRC_ALPHA;						// Only works when the bitmap contains an alpha channel

				// Update the window
				APIHelp.UpdateLayeredWindow(Handle, screenDc, ref newLocation, ref newSize,
					memDc, ref sourceLocation, 0, ref blend, APIHelp.ULW_ALPHA);
			}
			finally
			{
				// Release device context
				APIHelp.ReleaseDC(IntPtr.Zero, screenDc);
				if (hBitmap != IntPtr.Zero)
				{
					APIHelp.SelectObject(memDc, hOldBitmap);
					APIHelp.DeleteObject(hBitmap);										// Remove bitmap resources
				}
				APIHelp.DeleteDC(memDc);
			}
		}
		#endregion

		private void Form1_Load(object sender, EventArgs e)
		{
			PositionForm();

			showTimeZoneToolStripMenuItem.Checked = Properties.Settings.Default.ShowTimeZone;
			SystemInfo.ShowTimeZone = Properties.Settings.Default.ShowTimeZone;

			showComputerStatsToolStripMenuItem.Checked = Properties.Settings.Default.ShowComputerDetails;

			whiteToolStripMenuItem.Checked = false;
			yellowToolStripMenuItem.Checked = false;
			redToolStripMenuItem.Checked = false;
			blackToolStripMenuItem.Checked = false;

			if(Properties.Settings.Default.TextColor == Color.Red)
				redToolStripMenuItem.Checked = true;
			else if (Properties.Settings.Default.TextColor == Color.Yellow)
				yellowToolStripMenuItem.Checked = true;
			else if (Properties.Settings.Default.TextColor == Color.Black)
				blackToolStripMenuItem.Checked = true;
			else
				whiteToolStripMenuItem.Checked = true;

			notOftenToolStripMenuItem.Checked = false;
			reasonableToolStripMenuItem.Checked = false;
			reallyOftenToolStripMenuItem.Checked = false;
			superFastToolStripMenuItem.Checked = false;

			switch (Properties.Settings.Default.UpdateFreq)
			{
				case 2000:
					notOftenToolStripMenuItem.Checked = true;
					break;
				case 1000:
					reasonableToolStripMenuItem.Checked = true;
					break;
				case 500:
				default:
					reallyOftenToolStripMenuItem.Checked = true;
					break;
				case 250:
					superFastToolStripMenuItem.Checked = true;
					break;
			}
			timer1.Interval = Properties.Settings.Default.UpdateFreq;
			timer1.Start();

			m_started = true;
		}

		private void SendBehind()
		{
			APIHelp.SetWindowPos(this.Handle, (IntPtr)APIHelp.HWND_BOTTOM, 0, 0, 0, 0, APIHelp.SWP_NOMOVE | APIHelp.SWP_NOSIZE | APIHelp.SWP_SHOWWINDOW);
		}

		private void PositionForm()
		{
			if (Properties.Settings.Default.HOffset == 0.0 && Properties.Settings.Default.VOffset == 0.0)
			{
				this.Left = Screen.PrimaryScreen.Bounds.X + Screen.PrimaryScreen.Bounds.Width - m_screenOffset - this.Width;
				this.Top = m_screenOffset;
			}
			else
			{
				this.Left = Convert.ToInt32(Properties.Settings.Default.HOffset * Convert.ToDouble(Screen.PrimaryScreen.Bounds.Width));
				this.Top = Convert.ToInt32(Properties.Settings.Default.VOffset * Convert.ToDouble(Screen.PrimaryScreen.Bounds.Height));
			}
		}

		private void Form1_Move(object sender, EventArgs e)
		{
			if (m_started)
			{
				Properties.Settings.Default.HOffset = Convert.ToDouble(this.Left) / Convert.ToDouble(Screen.PrimaryScreen.Bounds.Width);
				Properties.Settings.Default.VOffset = Convert.ToDouble(this.Top) / Convert.ToDouble(Screen.PrimaryScreen.Bounds.Height);
				Properties.Settings.Default.Save();
			}
		}

		private void exitToolStripMenuItem_Click(object sender, EventArgs e)
		{
			Application.Exit();
		}

		private void timer1_Tick(object sender, EventArgs e)
		{
			UpdateInfo();
		}

		private void UpdateInfo()
		{
			try
			{
				using (Bitmap bm = new Bitmap(this.Width, this.Height))
				{
					using (Graphics g = Graphics.FromImage(bm))
					{
						using (Font f = new Font(m_font, m_fontSize))
						{
							using (GraphicsPath gPath = new GraphicsPath())
							{
								g.SmoothingMode = SmoothingMode.AntiAlias;
								g.TextRenderingHint = TextRenderingHint.AntiAlias;
								string result = "Computer: " + SystemInfo.MachineName + Environment.NewLine +
									"OS: " + SystemInfo.OSVersion + Environment.NewLine +
									"User: " + SystemInfo.UserDomain + Environment.NewLine +
									"IP Address: " + SystemInfo.IPAddress + Environment.NewLine +
									"Time: " + SystemInfo.Time + Environment.NewLine;
								if (Properties.Settings.Default.ShowComputerDetails)
									result += "CPU: " + SystemInfo.ProcessorUsage + Environment.NewLine +
										"Memory: " + SystemInfo.MemoryUsage + Environment.NewLine +
										"Disk: " + SystemInfo.DiskUsage + Environment.NewLine +
										"Process Count: " + SystemInfo.ProcessCount + Environment.NewLine;

								//Create Shadow path
								gPath.AddString(result, f.FontFamily, (int)f.Style, f.Size, new Rectangle(m_shadowDepth, m_shadowDepth, bm.Width - m_shadowDepth, bm.Height - m_shadowDepth), StringFormat.GenericDefault);

								//Draw Shadow
								using (SolidBrush shadowBrush = new SolidBrush(m_shadowColor))
									g.FillPath(shadowBrush, gPath);
								gPath.Reset();

								//Create the standard text path.
								gPath.AddString(result, f.FontFamily, (int)f.Style, f.Size, new Rectangle(0, 0, this.Width, this.Height), StringFormat.GenericDefault);

								using (SolidBrush textBrush = new SolidBrush(Properties.Settings.Default.TextColor))
									g.FillPath(textBrush, gPath);

								SelectBitmap(bm);
							}
						}
					}
				}
			}
			catch (Exception ex)
			{
				notifyIcon1.ShowBalloonTip(5000, "Update failed", ex.Message, ToolTipIcon.Error);
			}
		}

		private void Form1_Activated(object sender, EventArgs e)
		{
			SendBehind();
		}

		#region Icon's menu
		private void whiteToolStripMenuItem_Click(object sender, EventArgs e)
		{
			Properties.Settings.Default.TextColor = Color.White;
			Properties.Settings.Default.Save();
			whiteToolStripMenuItem.Checked = true;
			yellowToolStripMenuItem.Checked = false;
			redToolStripMenuItem.Checked = false;
			blackToolStripMenuItem.Checked = false;
			UpdateInfo();
		}

		private void yellowToolStripMenuItem_Click(object sender, EventArgs e)
		{
			Properties.Settings.Default.TextColor = Color.Yellow;
			Properties.Settings.Default.Save();
			whiteToolStripMenuItem.Checked = false;
			yellowToolStripMenuItem.Checked = true;
			redToolStripMenuItem.Checked = false;
			blackToolStripMenuItem.Checked = false;
			UpdateInfo();
		}

		private void redToolStripMenuItem_Click(object sender, EventArgs e)
		{
			Properties.Settings.Default.TextColor = Color.Red;
			Properties.Settings.Default.Save();
			whiteToolStripMenuItem.Checked = false;
			yellowToolStripMenuItem.Checked = false;
			redToolStripMenuItem.Checked = true;
			blackToolStripMenuItem.Checked = false;
			UpdateInfo();
		}

		private void blackToolStripMenuItem_Click(object sender, EventArgs e)
		{
			Properties.Settings.Default.TextColor = Color.Black;
			Properties.Settings.Default.Save();
			whiteToolStripMenuItem.Checked = false;
			yellowToolStripMenuItem.Checked = false;
			redToolStripMenuItem.Checked = false;
			blackToolStripMenuItem.Checked = true;
			UpdateInfo();
		}

		private void showTimeZoneToolStripMenuItem_Click(object sender, EventArgs e)
		{
			Properties.Settings.Default.ShowTimeZone = !Properties.Settings.Default.ShowTimeZone;
			Properties.Settings.Default.Save();
			showTimeZoneToolStripMenuItem.Checked = Properties.Settings.Default.ShowTimeZone;
			SystemInfo.ShowTimeZone = Properties.Settings.Default.ShowTimeZone;
			UpdateInfo();
		}

		private void showComputerStatsToolStripMenuItem_Click(object sender, EventArgs e)
		{
			Properties.Settings.Default.ShowComputerDetails = !Properties.Settings.Default.ShowComputerDetails;
			Properties.Settings.Default.Save();
			showComputerStatsToolStripMenuItem.Checked = Properties.Settings.Default.ShowComputerDetails;
			UpdateInfo();
		}

		private void notOftenToolStripMenuItem_Click(object sender, EventArgs e)
		{
			Properties.Settings.Default.UpdateFreq = 2000;
			Properties.Settings.Default.Save();
			notOftenToolStripMenuItem.Checked = true;
			reasonableToolStripMenuItem.Checked = false;
			reallyOftenToolStripMenuItem.Checked = false;
			superFastToolStripMenuItem.Checked = false;
			timer1.Interval = Properties.Settings.Default.UpdateFreq;
		}

		private void reasonableToolStripMenuItem_Click(object sender, EventArgs e)
		{
			Properties.Settings.Default.UpdateFreq = 1000;
			Properties.Settings.Default.Save();
			notOftenToolStripMenuItem.Checked = false;
			reasonableToolStripMenuItem.Checked = true;
			reallyOftenToolStripMenuItem.Checked = false;
			superFastToolStripMenuItem.Checked = false;
			timer1.Interval = Properties.Settings.Default.UpdateFreq;
		}

		private void reallyOftenToolStripMenuItem_Click(object sender, EventArgs e)
		{
			Properties.Settings.Default.UpdateFreq = 500;
			Properties.Settings.Default.Save();
			notOftenToolStripMenuItem.Checked = false;
			reasonableToolStripMenuItem.Checked = false;
			reallyOftenToolStripMenuItem.Checked = true;
			superFastToolStripMenuItem.Checked = false;
			timer1.Interval = Properties.Settings.Default.UpdateFreq;
		}

		private void superFastToolStripMenuItem_Click(object sender, EventArgs e)
		{
			Properties.Settings.Default.UpdateFreq = 250;
			Properties.Settings.Default.Save();
			notOftenToolStripMenuItem.Checked = false;
			reasonableToolStripMenuItem.Checked = false;
			reallyOftenToolStripMenuItem.Checked = false;
			superFastToolStripMenuItem.Checked = true;
			timer1.Interval = Properties.Settings.Default.UpdateFreq;
		}
		#endregion
	}
}