using System;
using System.Collections.Generic;
using System.Text;
using System.IO;

namespace GPS
{
	public class GPXWriter : IDisposable
	{
		private FileStream m_fs = null;
		private int m_pointsInTrack = 0;
		private int m_pointsInFile = 0;
		private DateTime m_lastPoint = DateTime.MinValue;

		public GPXWriter(string filename)
		{
			OpenFile(filename);
		}

		public int Points
		{
			get { return (m_pointsInFile); }
		}

		#region File Management
		private void OpenFile(string filename)
		{
			if (m_fs != null)
				return;

			if (File.Exists(filename))
			{
				// append
				m_fs = new FileStream(filename, FileMode.OpenOrCreate, FileAccess.ReadWrite);
				long offset = FindInsertionLocation();
				if (offset != -1)
					m_fs.Seek(offset, SeekOrigin.Begin);
			}
			else
			{
				m_fs = new FileStream(filename, FileMode.Create, FileAccess.ReadWrite);
				WriteHeader();
				WriteTrackHeader();
			}
			WriteTrackSegmentHeader();
			AdjustPosition();
		}

		private void CloseFile()
		{
			if (m_fs != null)
				m_fs.Close();
			m_fs = null;
		}

		private long FindInsertionLocation()
		{
			m_pointsInFile = 0;
			long len = m_fs.Seek(0, SeekOrigin.End);
			m_fs.Seek(0, SeekOrigin.Begin);
			byte[] data = new byte[len];
			m_fs.Read(data, 0, (int)len);
			string dataStr = Encoding.ASCII.GetString(data, 0, Convert.ToInt32(len));
			string[] points = dataStr.Split('<');
			foreach (string pt in points)
			{
				if (pt.StartsWith("trkpt"))
					m_pointsInFile++;
			}
			return ((long)dataStr.IndexOf("</trk>"));
		}

		private void AdjustPosition()
		{
			if (m_fs == null)
				return;
			// get offset where we can insert new data
			long offset = m_fs.Seek(0, SeekOrigin.Current);
			WriteTrackSegmentFooter();
			WriteTrackFooter();
			WriteFooter();
			// jump back to offset
			m_fs.Seek(offset, SeekOrigin.Begin);
			m_fs.Flush();
		}

		private void WriteLine(string line)
		{
			if (m_fs == null)
				return;
			//string tmp = line + Environment.NewLine;
			string tmp = line + "\r\n";
			byte[] data = Encoding.ASCII.GetBytes(tmp);
			m_fs.Write(data, 0, data.Length);
		}
		#endregion

		#region GPX File Data
		private void WriteHeader()
		{
			WriteLine("<?xml version=\"1.0\" standalone=\"yes\"?>");
			WriteLine("<gpx");
			WriteLine(" version=\"1.0\"");
			WriteLine(String.Format(" creator=\"{0} 1.0\"", HappyGPSLogger.Form1.ApplicationName));
			WriteLine(" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"");
			WriteLine(" xmlns=\"http://www.topografix.com/GPX/1/0\"");
			WriteLine(" xmlns:topografix=\"http://www.topografix.com/GPX/Private/TopoGrafix/0/2\"");
			WriteLine(" xsi:schemaLocation=\"http://www.topografix.com/GPX/1/0 http://www.topografix.com/GPX/1/0/gpx.xsd http://www.topografix.com/GPX/Private/TopoGrafix/0/2 http://www.topografix.com/GPX/Private/TopoGrafix/0/2/topografix.xsd\">");
			WriteLine(String.Format("<time>{0}</time>", DateTime.Now.ToUniversalTime().ToString("u").Replace(" ", "T")));
		}

		private void WriteFooter()
		{
			WriteLine("</gpx>");
		}

		private void WriteTrackHeader()
		{
			WriteLine("<trk>");
		}

		private void WriteTrackFooter()
		{
			WriteLine("</trk>");
		}

		private void WriteTrackSegmentHeader()
		{
			WriteLine("<trkseg>");
		}

		private void WriteTrackSegmentFooter()
		{
			WriteLine("</trkseg>");
		}

		private void WriteTrackPoint(string lat, string lon, string ele, string pdop)
		{
			WriteLine("<trkpt lat=\"" + lat + "\" lon=\"" + lon + "\">");
			WriteLine(" <ele>" + ele + "</ele>");
			WriteLine(String.Format(" <time>{0}</time>", DateTime.Now.ToUniversalTime().ToString("u").Replace(" ", "T")));
			WriteLine(" <pdop>" + pdop + "</pdop>");
			WriteLine("</trkpt>");
		}

		private void WriteTrackPoint(string lat, string lon, string ele)
		{
			WriteLine("<trkpt lat=\"" + lat + "\" lon=\"" + lon + "\">");
			WriteLine(" <ele>" + ele + "</ele>");
			WriteLine(String.Format(" <time>{0}</time>", DateTime.Now.ToUniversalTime().ToString("u").Replace(" ", "T")));
			WriteLine("</trkpt>");
		}

		private void WriteTrackPoint(string lat, string lon)
		{
			WriteLine("<trkpt lat=\"" + lat + "\" lon=\"" + lon + "\">");
			WriteLine(String.Format(" <time>{0}</time>", DateTime.Now.ToUniversalTime().ToString("u").Replace(" ", "T")));
			WriteLine("</trkpt>");
		}
		#endregion

		#region Public Methods
		public void SegmentBreak()
		{
			if (m_pointsInTrack > 0)
			{
				WriteTrackSegmentFooter();
				WriteTrackSegmentHeader();
				AdjustPosition();
			}
			m_pointsInTrack = 0;
			m_lastPoint = DateTime.MinValue;
		}

		public void AddPoint(double lat, double lon, double ele)
		{
			TimeSpan ts = DateTime.Now - m_lastPoint;
			if (ts.TotalSeconds < 1.0)
				return;

			if (ele == 0.0)
			{
				AddPoint(lat, lon);
			}
			else
			{
				WriteTrackPoint(lat.ToString(), lon.ToString(), ele.ToString());
				AdjustPosition();
				m_pointsInTrack++;
				m_pointsInFile++;
			}
		}

		public void AddPoint(double lat, double lon, double ele, double pdop)
		{
			TimeSpan ts = DateTime.Now - m_lastPoint;
			if (ts.TotalSeconds < 1.0)
				return;

			if (pdop == 0.0)
			{
				AddPoint(lat, lon, ele);
			}
			else
			{
				WriteTrackPoint(lat.ToString(), lon.ToString(), ele.ToString(), pdop.ToString());
				AdjustPosition();
				m_pointsInTrack++;
				m_pointsInFile++;
			}
		}

		public void AddPoint(double lat, double lon)
		{
			TimeSpan ts = DateTime.Now - m_lastPoint;
			if (ts.TotalSeconds < 1.0)
				return;

			WriteTrackPoint(lat.ToString(), lon.ToString());
			AdjustPosition();
			m_pointsInTrack++;
			m_pointsInFile++;
		}
		#endregion

		#region IDisposable Members
		public void Dispose()
		{
			CloseFile();
		}
		#endregion
	}
}
