using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Net;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Text;
using System.Threading;
using System.IO;
using System.Windows.Forms;
using GPS;

namespace HappyGPSLogger
{
	public partial class Form1 : Form
	{
		public static readonly string ApplicationName = "HappyGPSLogger";

		private NmeaInterpreter nmea = new NmeaInterpreter();
		private GPXWriter writer = null;
		private object m_lockObject = new object();
		private object m_lockObject2 = new object();

		private static string appDataPath = Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.ApplicationData), ApplicationName);

		private volatile bool m_run = false;

		private bool m_haveSignal = false;
		private double m_lastBearing = 0.0;
		private double m_lastAltitude = 0.0;
		private double m_lastLatitude = 0.0;
		private double m_lastLongitude = 0.0;
		private double m_lastSpeed = 0.0;
		private double m_currentSpeed = 0.0;
		private double m_lastPdop = 0.0;
		private DateTime m_lastUpdate = DateTime.MinValue;

		private readonly int m_distanceFactor = 3; // 3m per mph
		private readonly double m_minDistance = 0.005; // 5m
		private readonly double m_altitude = 5.0; // meters
		private readonly double m_speed = 10.0; // mph
		private readonly double m_bearing = 20.0; // degrees
		private readonly double m_duration = 60.0; // seconds
		private readonly int m_speedRange = 100; // mph

		private Bitmap m_speedometerFace = null;
		Rectangle speedometerBounds = Rectangle.Empty;

		private Bluetooth.BluetoothRadioMode m_bluetoothState = Bluetooth.BluetoothRadioMode.Off;

		public Form1()
		{
			InitializeComponent();
			this.Text = ApplicationName;

			m_bluetoothState = Bluetooth.Mode;
			Bluetooth.Mode = Bluetooth.BluetoothRadioMode.On;
		}

		private void OpenPort()
		{
			Thread.Sleep(500);
			if (serialPort1.IsOpen)
				return;
			menuItem8.Enabled = false;
			if (String.IsNullOrEmpty(Settings.BaudRate))
				serialPort1.BaudRate = 4800;
			else
				serialPort1.BaudRate = Convert.ToInt32(Settings.BaudRate);
			if (String.IsNullOrEmpty(Settings.SerialPort))
				serialPort1.PortName = "COM1";
			else
				serialPort1.PortName = Settings.SerialPort;
			serialPort1.Open();
			serialPort1.DiscardNull = false;
		}

		private void ClosePortThread()
		{
			serialPort1.Close();
		}

		private void ClosePort()
		{
			lock (m_lockObject2)
			{
				menuItem8.Enabled = true;
				if (serialPort1.IsOpen == false)
					return;
				Thread t = new Thread(new ThreadStart(ClosePortThread));
				t.Start();
				while (serialPort1.IsOpen)
				{
					Thread.Sleep(100);
					Application.DoEvents();
				}
			}
		}

		private void Form1_Load(object sender, EventArgs e)
		{
			Directory.CreateDirectory(appDataPath);

			nmea.FixLost += new NmeaInterpreter.FixLostEventHandler(nmea_FixLost);
			nmea.FixObtained += new NmeaInterpreter.FixObtainedEventHandler(nmea_FixObtained);
			nmea.PositionReceived += new NmeaInterpreter.PositionReceivedEventHandler(nmea_PositionReceived);
			nmea.SpeedReceived += new NmeaInterpreter.SpeedReceivedEventHandler(nmea_SpeedReceived);
			nmea.BearingReceived += new NmeaInterpreter.BearingReceivedEventHandler(nmea_BearingReceived);
			nmea.AltitudeReceived += new NmeaInterpreter.AltitudeReceivedEventhandler(nmea_AltitudeReceived);
			nmea.SatelliteCountReceived += new NmeaInterpreter.SatelliteCountReceivedEventhandler(nmea_SatelliteCountReceived);
			nmea.PDOPReceived += new NmeaInterpreter.PDOPReceivedEventHandler(nmea_PDOPReceived);

			menuItem3.Enabled = false;
			menuItem4.Enabled = false;

			// initialize speedometer
			lock (m_lockObject)
			{
				m_speedometerFace = new Bitmap(panel1.Width, panel1.Height);
				using (Graphics g = Graphics.FromImage(m_speedometerFace))
				{
					speedometerBounds = new Rectangle(0, 0, m_speedometerFace.Width, m_speedometerFace.Height);
					using (Brush b = new SolidBrush(Color.White))
					{
						g.FillRectangle(b, speedometerBounds);
					}
					speedometerBounds.Inflate(-1, -1);
					if (speedometerBounds.Height > speedometerBounds.Width)
						speedometerBounds.Height = speedometerBounds.Width;
					else
						speedometerBounds.Width = speedometerBounds.Height;
					using (Brush b = new SolidBrush(Color.LightGray))
						g.FillEllipse(b, speedometerBounds);
					using (Pen p = new Pen(Color.Black, 1))
						g.DrawEllipse(p, speedometerBounds);

					using (Font f = new Font("Arial", 8, FontStyle.Regular))
					{
						using (Brush b = new SolidBrush(Color.Black))
						{
							SizeF s = g.MeasureString("0", f);
							g.DrawString("0", f, b, speedometerBounds.Width / 2 - s.Width / 2, speedometerBounds.Height - s.Height - 5);
							string label = (0.25 * m_speedRange).ToString();
							s = g.MeasureString(label, f);
							g.DrawString(label, f, b, 5, speedometerBounds.Height / 2 - s.Height / 2);
							label = (0.5 * m_speedRange).ToString();
							s = g.MeasureString(label, f);
							g.DrawString(label, f, b, speedometerBounds.Width / 2 - s.Width / 2, 5);
							label = (0.75 * m_speedRange).ToString();
							s = g.MeasureString(label, f);
							g.DrawString(label, f, b, speedometerBounds.Width - s.Width - 5, speedometerBounds.Height / 2 - s.Height / 2);
							label = "mph";
							s = g.MeasureString(label, f);
							g.DrawString(label, f, b, speedometerBounds.Width / 2 - s.Width / 2, speedometerBounds.Height / 2 - s.Height / 2);
						}
					}
				}
			}
			UpdateSpeedometer();
		}

		private void menuItem9_Click(object sender, EventArgs e)
		{
			lock (m_lockObject)
			{
				if (m_speedometerFace != null)
				{
					m_speedometerFace.Dispose();
					m_speedometerFace = null;
				}
			}
			menuItem3_Click(null, null);
			Bluetooth.Mode = m_bluetoothState;
			Application.Exit();
		}

		private void menuItem8_Click(object sender, EventArgs e)
		{
			Settings s = new Settings();
			if (s.ShowDialog() == DialogResult.OK)
			{
				//Form1_Load(null, null);
			}
		}

		#region Logging
		#region Thread-Safe functions
		private delegate void UpdateSpeedometerCallback();
		private void UpdateSpeedometer()
		{
			if (panel1.InvokeRequired)
			{
				this.Invoke(new UpdateSpeedometerCallback(UpdateSpeedometer));
			}
			else
			{
				panel1.Invalidate();
				//Application.DoEvents();
			}
		}

		private delegate void SetTextCallback(int label, string text);
		private void SetText(int label, string text)
		{
			if (label1.InvokeRequired)
			{
				this.Invoke(new SetTextCallback(SetText), new object[] { label, text });
			}
			else
			{
				switch (label)
				{
					case 1:
					default:
						label1.Text = text;
						break;
					case 2:
						label2.Text = text;
						break;
					case 3:
						label3.Text = text;
						break;
					case 4:
						label4.Text = text;
						break;
					case 5:
						label5.Text = text;
						break;
				}

				//Application.DoEvents();
			}
		}
		#endregion

		private void UpdatePointCount()
		{
			if(writer != null)
				SetText(3, "{" + writer.Points + "}");
		}

		private string LogFileName
		{
			get
			{
				string file = "Log-" + DateTime.Now.ToShortDateString().Replace("/", "-") + ".gpx";
				if (Directory.Exists(@"\Storage Card"))
					return (Path.Combine(@"\Storage Card", file));
				else
					return (Path.Combine(appDataPath, file));
			}
		}

		void nmea_PDOPReceived(double value)
		{
			m_lastPdop = value;
		}

		void nmea_SatelliteCountReceived(int value)
		{
			if (m_haveSignal)
				SetText(2, String.Format("Tracking {0} satellites.", value));
		}

		void nmea_AltitudeReceived(double value)
		{
			if (Math.Abs(m_lastAltitude - value) > m_altitude)
			{
				m_lastAltitude = value;
				if (writer != null)
				{
					writer.AddPoint(m_lastLatitude, m_lastLongitude, m_lastAltitude, m_lastPdop);
				}
			}
			UpdatePointCount();
		}

		void nmea_BearingReceived(double bearing)
		{
			if (Math.Abs(m_lastBearing - bearing) > m_bearing)
			{
				m_lastBearing = bearing;
				if (writer != null)
				{
					writer.AddPoint(m_lastLatitude, m_lastLongitude, m_lastAltitude, m_lastPdop);
				}
			}
			UpdatePointCount();
		}

		void nmea_SpeedReceived(double speed)
		{
			SetText(4, String.Format("{0} mph", Convert.ToInt32(speed)));
			lock (m_lockObject)
				m_currentSpeed = speed;
			UpdateSpeedometer();
			if (Math.Abs(m_lastSpeed - speed) > m_speed)
			{
				m_lastSpeed = speed;
				if (writer != null)
				{
					writer.AddPoint(m_lastLatitude, m_lastLongitude, m_lastAltitude, m_lastPdop);
				}
			}
			UpdatePointCount();
		}

		void nmea_PositionReceived(string latitude, string longitude)
		{
			//Distance between on degree of longitude at a given latitude:
			//(pi/180)*R*cosA where R is the radius of the earth (3963.1676mi) and A is the degree latitude

			SetText(1, String.Format("Lat: {0}\r\nLng: {1}", latitude, longitude));
			SetText(5, String.Format("Last Update: {0}", DateTime.Now.ToString("G")));
			double lat = Degrees.ConvertDegrees(latitude);
			double lon = Degrees.ConvertDegrees(longitude);
			double dist = DistanceBetweenPoints.Calc(lat, lon, m_lastLatitude, m_lastLongitude);
			TimeSpan ts = DateTime.Now - m_lastUpdate;
			double distance = (Convert.ToDouble(m_distanceFactor) * m_lastSpeed) / 1000.0;
			if (distance < m_minDistance)
				distance = m_minDistance;
			if (dist > distance || ts.TotalSeconds > m_duration)
			{
				m_lastLatitude = lat;
				m_lastLongitude = lon;
				m_lastUpdate = DateTime.Now;
				if (writer != null)
				{
					writer.AddPoint(m_lastLatitude, m_lastLongitude, m_lastAltitude, m_lastPdop);
				}
			}
			UpdatePointCount();
		}

		void nmea_FixObtained()
		{
			m_haveSignal = true;
		}

		void nmea_FixLost()
		{
			m_haveSignal = false;
			SetText(2, "Signal Lost");
			if (writer != null)
				writer.SegmentBreak();
		}
		
		void ParseLine(string line)
		{
			if (String.IsNullOrEmpty(line))
				return;
			string input = line.Trim();
			nmea.Parse(input);
		}

		private void menuItem2_Click(object sender, EventArgs e)
		{
			menuItem2.Enabled = false;
			menuItem3.Enabled = true;
			menuItem4.Enabled = true;

			try
			{
				OpenPort();
				writer = new GPXWriter(LogFileName);
				m_run = true;
				UpdatePointCount();
			}
			catch (Exception ex)
			{
				MessageBox.Show("Error logging positon: " + ex.Message, "Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation, MessageBoxDefaultButton.Button1);
				m_run = false;
			}
			UpdateSpeedometer();
		}

		private void menuItem3_Click(object sender, EventArgs e)
		{
			m_run = false;
			menuItem2.Enabled = true;
			menuItem3.Enabled = false;
			menuItem4.Enabled = false;

			SetText(1, "Log ended.");
			SetText(2, "");
			SetText(3, "");
			SetText(4, "");

			if (writer != null)
			{
				writer.Dispose();
				writer = null;
			}
			UpdateSpeedometer();

			ClosePort();
		}
		#endregion

		private void serialPort1_DataReceived(object sender, System.IO.Ports.SerialDataReceivedEventArgs e)
		{
			if (m_run)
			{
				try
				{
					string line = String.Empty;
					lock (m_lockObject2)
					{
						if(m_run)
							line = serialPort1.ReadLine();
					}
					if(!String.IsNullOrEmpty(line))
						ParseLine(line);
				}
				catch (TimeoutException) { }
				catch (Exception ex)
				{
					SetText(5, ex.Message);
				}
			}
		}

		private void menuItem4_Click(object sender, EventArgs e)
		{
			m_lastUpdate = DateTime.Now;
			if (writer != null)
				writer.AddPoint(m_lastLatitude, m_lastLongitude, m_lastAltitude, m_lastPdop);
			UpdatePointCount();
		}

		private void Form1_KeyDown(object sender, KeyEventArgs e)
		{
			if ((e.KeyCode == System.Windows.Forms.Keys.Up) || (e.KeyCode == System.Windows.Forms.Keys.Down))
			{
				// add point
				m_lastUpdate = DateTime.Now;
				if (writer != null)
					writer.AddPoint(m_lastLatitude, m_lastLongitude, m_lastAltitude, m_lastPdop);
				UpdatePointCount();
			}
			if ((e.KeyCode == System.Windows.Forms.Keys.Enter))
			{
				if (writer == null)
					menuItem2_Click(null, null);
				else
					menuItem3_Click(null, null);
			}
			if ((e.KeyCode == System.Windows.Forms.Keys.D0))
			{
				// 0
				menuItem9_Click(null, null);
			}
		}

		private void panel1_Paint(object sender, PaintEventArgs e)
		{
			lock (m_lockObject)
			{
				int speed = Convert.ToInt32(m_currentSpeed);
				if (speed > m_speedRange)
					speed = m_speedRange;
				if (m_speedometerFace == null)
					return;


				if (m_run == false)
				{
					using (Brush b = new SolidBrush(Color.White))
						e.Graphics.FillRectangle(b, speedometerBounds);
				}
				else
				{
					using (Pen p = new Pen(Color.Red, 2))
					{
						e.Graphics.DrawImage(m_speedometerFace, 0, 0);
						double angle = 2.0 * Math.PI * (Convert.ToDouble(speed) - (Convert.ToDouble(speed) / Convert.ToDouble(m_speedRange)));
						double len = speedometerBounds.Width / 2.0 * 0.8;
						int hOffset = 1 + speedometerBounds.Width / 2;
						int vOffset = 1 + speedometerBounds.Height / 2;
						e.Graphics.DrawLine(p, hOffset, vOffset, hOffset + Convert.ToInt32(Math.Sin(angle) * len), vOffset + Convert.ToInt32(Math.Cos(angle) * len));
					}
				}
			}
		}
	}
}